<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/Profile.php';
require_once __DIR__ . '/../middleware/jwtMiddleware.php';

class ProfileController {
    private $profileModel;
    
    public function __construct() {
        $database = new Database();
        $db = $database->getConnection();
        $this->profileModel = new Profile($db);
        error_log("✅ ProfileController initialized");
    }

    public function updateBasicInfo() {
        error_log("🟢 Starting basic info update...");
        
        try {
            // Get user ID from JWT middleware
            $user_id = jwtMiddleware();
            
            // Get input data
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid input data']);
                return;
            }

            error_log("📝 Basic info update data received for user $user_id: " . print_r($input, true));

            // Validate required fields
            if (!isset($input['name']) || empty(trim($input['name']))) {
                http_response_code(400);
                echo json_encode(['error' => 'Name is required']);
                return;
            }

            // Prepare update data
            $update_data = [
                'name' => trim($input['name']),
                'nickname' => isset($input['nickname']) ? trim($input['nickname']) : null,
                'gender' => isset($input['gender']) ? trim($input['gender']) : null,
                'age' => isset($input['age']) ? (int)$input['age'] : null,
                'location' => isset($input['location']) ? trim($input['location']) : null,
                'bio' => isset($input['bio']) ? trim($input['bio']) : null,
                'preference' => isset($input['preference']) ? trim($input['preference']) : null,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            // Validate gender
            if ($update_data['gender'] && !in_array($update_data['gender'], ['Male', 'Female'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid gender value']);
                return;
            }

            // Validate age
            if ($update_data['age'] && ($update_data['age'] < 18 || $update_data['age'] > 100)) {
                http_response_code(400);
                echo json_encode(['error' => 'Age must be between 18 and 100']);
                return;
            }

            // Validate preference
            if ($update_data['preference'] && !in_array($update_data['preference'], ['Men', 'Women', 'Both'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid preference value']);
                return;
            }

            // Update profile
            $result = $this->profileModel->updateUserProfile($user_id, $update_data);
            
            if ($result) {
                error_log("✅ Basic info updated successfully for user: $user_id");
                
                // Get updated user data
                $updated_user = $this->profileModel->getUserById($user_id);
                
                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'message' => 'Profile updated successfully',
                    'user' => $updated_user
                ]);
            } else {
                throw new Exception('Failed to update profile in database');
            }

        } catch (Exception $e) {
            error_log("❌ Error updating basic info: " . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Failed to update profile',
                'message' => $e->getMessage()
            ]);
        }
    }

    public function updatePassword() {
        error_log("🟢 Starting password update...");
        
        try {
            $user_id = jwtMiddleware();
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!$input || !isset($input['current_password']) || !isset($input['new_password'])) {
                http_response_code(400);
                echo json_encode(['error' => 'Current password and new password are required']);
                return;
            }

            $current_password = trim($input['current_password']);
            $new_password = trim($input['new_password']);

            // Validate new password
            if (strlen($new_password) < 6) {
                http_response_code(400);
                echo json_encode(['error' => 'New password must be at least 6 characters long']);
                return;
            }

            // Update password
            $result = $this->profileModel->updateUserPassword($user_id, $current_password, $new_password);
            
            if ($result) {
                error_log("✅ Password updated successfully for user: $user_id");
                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'message' => 'Password updated successfully'
                ]);
            } else {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'error' => 'Current password is incorrect'
                ]);
            }

        } catch (Exception $e) {
            error_log("❌ Error updating password: " . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Failed to update password',
                'message' => $e->getMessage()
            ]);
        }
    }

    public function getProfile() {
        error_log("🟢 Getting user profile...");
        
        try {
            $user_id = jwtMiddleware();
            $user = $this->profileModel->getUserById($user_id);
            
            if ($user) {
                error_log("✅ Profile retrieved for user: $user_id");
                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'user' => $user
                ]);
            } else {
                http_response_code(404);
                echo json_encode([
                    'success' => false,
                    'error' => 'User not found'
                ]);
            }

        } catch (Exception $e) {
            error_log("❌ Error getting profile: " . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Failed to get profile',
                'message' => $e->getMessage()
            ]);
        }
    }
}
?>