<?php
// Headers must be FIRST, before any output
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, PUT, OPTIONS, DELETE");
header("Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With, Accept, Origin");
header("Access-Control-Expose-Headers: Content-Length, X-JSON");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Include logger at the very beginning
require_once __DIR__ . '/../helpers/logger.php';

Logger::log("🚀 ============ DISCOVERY ENDPOINT CALLED ============");
Logger::log("🌐 Request Details", [
    'method' => $_SERVER['REQUEST_METHOD'],
    'uri' => $_SERVER['REQUEST_URI'],
    'full_url' => (isset($_SERVER['HTTPS']) ? 'https' : 'http') . '://' . 
                  ($_SERVER['HTTP_HOST'] ?? '') . 
                  ($_SERVER['REQUEST_URI'] ?? '')
]);

try {
    require_once __DIR__ . '/../config/database.php';
    require_once __DIR__ . '/../controllers/DiscoveryController.php';
    
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    
    Logger::log("📡 Routing Info", [
        'method' => $method,
        'action' => $action,
        'all_get_params' => $_GET,
        'server_request_method' => $_SERVER['REQUEST_METHOD'],
        'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'N/A'
    ]);
    
    $discoveryController = new DiscoveryController();
    
    // Log controller creation
    Logger::log("🎯 DiscoveryController instantiated");
    
    // Handle both singular and plural actions for now
    $discoverActions = ['discover-users', 'discover-user'];
    
    switch(true) {
        case ($method == 'GET' && in_array($action, $discoverActions)):
            Logger::log("📍 ROUTING: GET discover users (action: $action)");
            $response = $discoveryController->getDiscoverUsers();
            Logger::logResponse($response ?? 'No response from controller');
            break;
            
        case ($method == 'POST' && $action == 'handle-swipe'):
            Logger::log("📍 ROUTING: POST handle swipe");
            $response = $discoveryController->handleSwipe();
            Logger::logResponse($response ?? 'No response from controller');
            break;
            
        case ($method == 'POST' && $action == 'update-location'):
            Logger::log("📍 ROUTING: POST update location");
            $response = $discoveryController->updateLocation();
            Logger::logResponse($response ?? 'No response from controller');
            break;
            
        case ($method == 'GET' && $action == 'matches'):
            Logger::log("📍 ROUTING: GET matches");
            $response = $discoveryController->getMatches();
            Logger::logResponse($response ?? 'No response from controller');
            break;
            
        case ($method == 'GET' && $action == 'debug'):
            Logger::log("📍 ROUTING: GET debug");
            $response = $discoveryController->debug();
            Logger::logResponse($response ?? 'No response from controller');
            break;
            
        case ($method == 'GET' && $action == 'ping'):
            Logger::log("📍 ROUTING: GET ping (test endpoint)");
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'message' => 'Discovery API is live',
                'timestamp' => time(),
                'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
                'logs' => Logger::getLogFile()
            ]);
            break;
            
        case ($method == 'GET' && $action == 'logs'):
            // Securely show recent logs (only for development!)
            if ($_SERVER['REMOTE_ADDR'] === '127.0.0.1' || $_SERVER['REMOTE_ADDR'] === '::1') {
                $logFile = Logger::getLogFile();
                if (file_exists($logFile)) {
                    $logs = file_get_contents($logFile);
                    echo "<pre>" . htmlspecialchars($logs) . "</pre>";
                } else {
                    echo "No log file found";
                }
            } else {
                http_response_code(403);
                echo json_encode(['error' => 'Access denied']);
            }
            break;
            
        default:
            Logger::log("❌ ROUTING ERROR: No matching route found", [
                'method' => $method,
                'action' => $action,
                'expected_actions' => array_merge($discoverActions, [
                    'handle-swipe',
                    'update-location',
                    'matches',
                    'debug',
                    'ping'
                ])
            ]);
            
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'error' => 'Discovery route not found',
                'debug' => [
                    'received_method' => $method,
                    'received_action' => $action,
                    'is_singular' => $action === 'discover-user',
                    'is_plural' => $action === 'discover-users',
                    'all_get_params' => $_GET,
                    'query_string' => $_SERVER['QUERY_STRING'] ?? '',
                    'request_uri' => $_SERVER['REQUEST_URI'],
                    'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
                    'timestamp' => date('Y-m-d H:i:s')
                ],
                'available_routes' => [
                    'GET discovery.php?action=discover-users (or discover-user)',
                    'POST discovery.php?action=handle-swipe',
                    'POST discovery.php?action=update-location',
                    'GET discovery.php?action=matches',
                    'GET discovery.php?action=debug',
                    'GET discovery.php?action=ping (test endpoint)'
                ],
                'note' => 'Both singular (discover-user) and plural (discover-users) are now accepted'
            ]);
            break;
    }
    
} catch (Exception $e) {
    Logger::logError("💥 Unhandled exception in discovery route", $e);
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Internal server error',
        'message' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine(),
        'request_id' => $_SERVER['REQUEST_ID'] ?? 'N/A',
        'trace' => $e->getTraceAsString()
    ]);
}

Logger::log("✅ ============ DISCOVERY REQUEST COMPLETE ============\n\n");