<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/logger.php';

class Discovery {
    private $db;
    private $users_table = "users";
    private $photos_table = "user_photos";
    private $swipes_table = "user_swipes";
    private $matches_table = "user_matches";

    public function __construct($db) {
        $this->db = $db;
        //$this->db->exec("SET NAMES 'utf16'");
        //$this->db->exec("SET CHARACTER SET utf16");
    }

    public function getDiscoveryUsers($current_user_id, $current_user, $offset, $limit) {
    try {
        Logger::log("🔍 getDiscoveryUsers called", [
            'current_user_id' => $current_user_id,
            'current_user_gender' => $current_user['gender'] ?? null,
            'current_user_preference' => $current_user['preference'] ?? null
        ]);
        
        // SIMPLIFIED QUERY - Get all users except current
        $query = "SELECT 
            u.id, u.name, u.age, u.bio, u.location, u.gender, u.preference,
            u.created_at, u.last_active, u.latitude, u.longitude,
            
            -- Get a random photo
            (SELECT photo_url FROM " . $this->photos_table . " 
             WHERE user_id = u.id 
             ORDER BY RAND() LIMIT 1) as photo_url,
            
            -- Count total photos
            (SELECT COUNT(*) FROM " . $this->photos_table . " 
             WHERE user_id = u.id) as total_photos
                
        FROM " . $this->users_table . " u
        WHERE u.id != :current_user_id
        AND u.profile_complete = 1
    
        LIMIT :limit OFFSET :offset";
        
        Logger::log("📝 Query:", $query);
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->execute();
        
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        Logger::log("✅ Database returned " . count($users) . " users");
        
        if (count($users) > 0) {
            Logger::log("👤 Sample user from DB:", [
                'id' => $users[0]['id'],
                'name' => $users[0]['name'],
                'gender' => $users[0]['gender'],
                'preference' => $users[0]['preference'],
                'photo_url' => $users[0]['photo_url'] ?? 'none',
                'total_photos' => $users[0]['total_photos']
            ]);
        }
        
        // Now filter by preferences IN PHP (simpler)
        $filtered_users = [];
        $current_user_gender = $current_user['gender'] ?? null;
        $current_user_preference = $current_user['preference'] ?? 'Both';
        
        Logger::log("🔍 Filtering preferences:", [
            'current_gender' => $current_user_gender,
            'current_preference' => $current_user_preference
        ]);
        
        foreach ($users as $user) {
            // Skip users without photos
            if (empty($user['photo_url'])) {
                Logger::log("⚠️ Skipping user {$user['id']} - no photos");
                continue;
            }
            
            // Apply gender preference filtering
            if ($this->usersMatchPreferences($current_user_gender, $current_user_preference, $user)) {
                $filtered_users[] = $user;
            } else {
                Logger::log("⚠️ Skipping user {$user['id']} - preferences don't match");
            }
        }
        
        Logger::log("✅ After filtering: " . count($filtered_users) . " users");
        
        // Format the response
        foreach ($filtered_users as &$user) {
            $user['pics'] = [];
            if (!empty($user['photo_url'])) {
                $user['pics'][] = $user['photo_url'];
            }
            unset($user['photo_url']);
            
            // Simple distance display
            $user['distance'] = $user['location'] ?? 'Nearby';
            
            // Active status calculation
            if (!empty($user['last_active'])) {
                $last_active = strtotime($user['last_active']);
                $now = time();
                $diff = $now - $last_active;
                
                if ($diff < 300) { // 5 minutes
                    $user['active_status'] = 'Online';
                } elseif ($diff < 3600) {
                    $minutes = floor($diff / 60);
                    $user['active_status'] = $minutes . 'm ago';
                } elseif ($diff < 86400) {
                    $hours = floor($diff / 3600);
                    $user['active_status'] = $hours . 'h ago';
                } else {
                    $user['active_status'] = floor($diff / 86400) . 'd ago';
                }
            } else {
                $user['active_status'] = 'Recently';
            }
        }
        
        return $filtered_users;
        
    } catch (Exception $e) {
        Logger::logError("❌ Error in getDiscoveryUsers", $e);
        return [];
    }
}

// Simple preference matching in PHP
private function usersMatchPreferences($current_gender, $current_preference, $target_user) {
    // If no preferences set, match everyone
    if (!$current_gender || !$current_preference) {
        return true;
    }
    
    $target_gender = $target_user['gender'] ?? null;
    $target_preference = $target_user['preference'] ?? 'Both';
    
    // 1. Check if current user wants this gender
    $current_wants_target = true;
    if ($current_preference === 'Men' && $target_gender !== 'Male') {
        $current_wants_target = false;
    } elseif ($current_preference === 'Women' && $target_gender !== 'Female') {
        $current_wants_target = false;
    }
    // 'Both' means wants everyone, so always true
    
    // 2. Check if target user wants current user's gender
    $target_wants_current = true;
    if ($target_preference === 'Men' && $current_gender !== 'Male') {
        $target_wants_current = false;
    } elseif ($target_preference === 'Women' && $current_gender !== 'Female') {
        $target_wants_current = false;
    }
    // 'Both' means wants everyone, so always true
    
    Logger::log("🤝 Preference check:", [
        'current_gender' => $current_gender,
        'current_pref' => $current_preference,
        'target_gender' => $target_gender,
        'target_pref' => $target_preference,
        'current_wants_target' => $current_wants_target,
        'target_wants_current' => $target_wants_current,
        'match' => ($current_wants_target && $target_wants_current)
    ]);
    
    return $current_wants_target && $target_wants_current;
}

// Simplified count method
public function getDiscoveryUsersCount($current_user_id, $current_user) {
    try {
        Logger::log("🔍 getDiscoveryUsersCount called");
        
        // Simple count - we'll filter in PHP
        $query = "SELECT COUNT(*) as total
                  FROM " . $this->users_table . " u
                  WHERE u.id != :current_user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':current_user_id', $current_user_id, PDO::PARAM_INT);
        $stmt->execute();
        
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        Logger::log("✅ Total users in DB (excluding current): " . ($result['total'] ?? 0));
        
        // This is an approximation - actual filtered count might be less
        return $result['total'] ?? 0;
        
    } catch (Exception $e) {
        Logger::logError("Error in getDiscoveryUsersCount", $e);
        return 0;
    }
}

// Helper method to build gender preference conditions
private function buildGenderPreferenceConditions($current_gender, $current_preference) {
    if (!$current_gender || !$current_preference) {
        return '';
    }
    
    $conditions = [];
    
    // Current user's preference
    switch ($current_preference) {
        case 'Men':
            $conditions[] = "u.gender = 'Male'";
            break;
        case 'Women':
            $conditions[] = "u.gender = 'Female'";
            break;
        case 'Both':
            // No restriction on target gender
            break;
    }
    
    // Target user's preference should match current user's gender
    $conditions[] = "(u.preference = 'Both' OR 
                     (u.preference = 'Men' AND :current_gender = 'Male') OR 
                     (u.preference = 'Women' AND :current_gender = 'Female'))";
    
    return implode(' AND ', $conditions);
}

// Helper method to calculate distance using Haversine formula
private function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371; // in kilometers
    
    $latFrom = deg2rad($lat1);
    $lonFrom = deg2rad($lon1);
    $latTo = deg2rad($lat2);
    $lonTo = deg2rad($lon2);
    
    $latDelta = $latTo - $latFrom;
    $lonDelta = $lonTo - $lonFrom;
    
    $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) +
        cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
    
    return $angle * $earthRadius;
}


public function getCurrentUser($user_id) {
    try {
        $query = "SELECT * FROM " . $this->users_table . " WHERE id = :id";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':id', $user_id, PDO::PARAM_INT);
        $stmt->execute();
        
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        Logger::logError("Error in getCurrentUser", $e);
        return false;
    }
}

    public function recordSwipe($user_id, $target_user_id, $action) {
        try {
            // Check if swipe already exists
            $check_query = "SELECT id FROM " . $this->swipes_table . " 
                           WHERE user_id = :user_id AND target_user_id = :target_user_id";
            $check_stmt = $this->db->prepare($check_query);
            $check_stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $check_stmt->bindParam(':target_user_id', $target_user_id, PDO::PARAM_INT);
            $check_stmt->execute();
            
            if ($check_stmt->rowCount() > 0) {
                $query = "UPDATE " . $this->swipes_table . " 
                         SET action = :action, updated_at = NOW() 
                         WHERE user_id = :user_id AND target_user_id = :target_user_id";
            } else {
                $query = "INSERT INTO " . $this->swipes_table . " 
                         (user_id, target_user_id, action, created_at, updated_at) 
                         VALUES (:user_id, :target_user_id, :action, NOW(), NOW())";
            }
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->bindParam(':target_user_id', $target_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':action', $action);
            
            return $stmt->execute();
            
        } catch (Exception $e) {
            Logger::logError("Error recording swipe", $e);
            return false;
        }
    }

    public function checkForMatch($user_id, $target_user_id) {
        try {
            $query = "SELECT id FROM " . $this->swipes_table . " 
                     WHERE user_id = :target_user_id 
                     AND target_user_id = :user_id 
                     AND action = 'like'";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':target_user_id', $target_user_id, PDO::PARAM_INT);
            $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->execute();
            
            if ($stmt->rowCount() > 0) {
                $match_query = "INSERT INTO " . $this->matches_table . " 
                              (user1_id, user2_id, created_at, updated_at) 
                              VALUES (:user1_id, :user2_id, NOW(), NOW())
                              ON DUPLICATE KEY UPDATE updated_at = NOW()";
                
                $match_stmt = $this->db->prepare($match_query);
                $user1_id = min($user_id, $target_user_id);
                $user2_id = max($user_id, $target_user_id);
                
                $match_stmt->bindParam(':user1_id', $user1_id, PDO::PARAM_INT);
                $match_stmt->bindParam(':user2_id', $user2_id, PDO::PARAM_INT);
                $match_stmt->execute();
                
                return true;
            }
            
            return false;
            
        } catch (Exception $e) {
            Logger::logError("Error checking for match", $e);
            return false;
        }
    }

    public function updateUserLocation($user_id, $latitude, $longitude) {
        try {
            $query = "UPDATE " . $this->users_table . " 
                     SET latitude = :latitude, longitude = :longitude, 
                         last_active = NOW(), updated_at = NOW() 
                     WHERE id = :user_id";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':latitude', $latitude);
            $stmt->bindParam(':longitude', $longitude);
            $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            
            return $stmt->execute();
            
        } catch (Exception $e) {
            Logger::logError("Error updating location", $e);
            return false;
        }
    }

    public function updateLastActive($user_id) {
        try {
            $query = "UPDATE " . $this->users_table . " 
                     SET last_active = NOW() 
                     WHERE id = :user_id";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            
            return $stmt->execute();
            
        } catch (Exception $e) {
            Logger::logError("Error updating last active", $e);
            return false;
        }
    }

    public function getUserMatches($user_id) {
        try {
            $query = "SELECT 
                u.id, u.name, u.age, u.location,
                (SELECT photo_url FROM " . $this->photos_table . " 
                 WHERE user_id = u.id AND is_main = 1 LIMIT 1) as photo_url,
                m.created_at as matched_at
            FROM " . $this->matches_table . " m
            JOIN " . $this->users_table . " u ON (
                (m.user1_id = :user_id AND m.user2_id = u.id) OR
                (m.user2_id = :user_id AND m.user1_id = u.id)
            )
            WHERE (m.user1_id = :user_id OR m.user2_id = :user_id)
            ORDER BY m.created_at DESC";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            Logger::logError("Error getting matches", $e);
            return [];
        }
    }
}