<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';

class Auth {
    private $db;
    private $table_name = "users";

    public function __construct($db) {
        $this->db = $db;
        error_log("✅ Auth model initialized with table: {$this->table_name}");
    }

    public function emailExists($email) {
        $query = "SELECT id, name, email, password, nickname, gender, age, location, bio, preference 
                  FROM " . $this->table_name . " 
                  WHERE email = :email 
                  LIMIT 1";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':email', $email);
        
        if ($stmt->execute()) {
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            return $user;
        }
        
        return false;
    }

    public function createUser($name, $email, $password = null, $googleId = null, $avatarUrl = null, $nickname = null, $gender = null, $age = null, $location = null, $bio = null, $preference = null, $latitude = null, $longitude = null) {
        $hashed_password = $password ? password_hash($password, PASSWORD_DEFAULT) : null;
        
        $query = "INSERT INTO " . $this->table_name . " 
                 (name, email, password, google_id, avatar_url, nickname, gender, age, location, bio, preference, latitude, longitude, created_at, last_active) 
                 VALUES (:name, :email, :password, :google_id, :avatar_url, :nickname, :gender, :age, :location, :bio, :preference, :latitude, :longitude, NOW(), NOW())";
        
        $stmt = $this->db->prepare($query);
        
        // Bind parameters
        $stmt->bindParam(':name', $name);
        $stmt->bindParam(':email', $email);
        $stmt->bindParam(':password', $hashed_password);
        $stmt->bindParam(':google_id', $googleId);
        $stmt->bindParam(':avatar_url', $avatarUrl);
        $stmt->bindParam(':nickname', $nickname);
        $stmt->bindParam(':gender', $gender);
        $stmt->bindParam(':age', $age);
        $stmt->bindParam(':location', $location);
        $stmt->bindParam(':bio', $bio);
        $stmt->bindParam(':preference', $preference);
        $stmt->bindParam(':latitude', $latitude);
        $stmt->bindParam(':longitude', $longitude);
        
        try {
            $result = $stmt->execute();
            if ($result) {
                $userId = $this->db->lastInsertId();
                error_log("✅ User created successfully with ID: $userId");
                return $userId;
            } else {
                error_log("❌ User creation failed");
                return false;
            }
        } catch (PDOException $e) {
            error_log("❌ User creation PDO Exception: " . $e->getMessage());
            return false;
        }
    }

    public function getUserById($id) {
        $query = "SELECT id, name, email, nickname, gender, age, location, bio, preference, profile_picture, avatar_url, latitude, longitude 
                 FROM " . $this->table_name . " 
                 WHERE id = :id 
                 LIMIT 1";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':id', $id);
        
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        return false;
    }

    public function getUserProfile($userId) {
        $query = "SELECT id, name, email, nickname, gender, age, location, bio, preference, profile_picture, avatar_url, latitude, longitude, created_at
                 FROM " . $this->table_name . " 
                 WHERE id = :id 
                 LIMIT 1";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':id', $userId);
        
        if ($stmt->execute()) {
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        return false;
    }
}
?>