<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../models/Discovery.php';
require_once __DIR__ . '/../middleware/jwtMiddleware.php';
require_once __DIR__ . '/../helpers/logger.php';  // Add this line

class DiscoveryController {
    private $discoveryModel;
    private $db;
    
    public function __construct() {
        Logger::log("🎯 DiscoveryController constructor called");
        
        $database = new Database();
        $this->db = $database->getConnection();
        $this->discoveryModel = new Discovery($this->db);
        
        Logger::log("✅ DiscoveryController initialized", [
            'db_connected' => $this->db !== null,
            'model_initialized' => $this->discoveryModel !== null
        ]);
    }

    public function getDiscoverUsers() {
    Logger::log("🟢 getDiscoverUsers() method called");
    
    try {
        Logger::log("🔐 Calling jwtMiddleware...");
        $user_id = jwtMiddleware();
        Logger::log("🔐 jwtMiddleware returned", ['user_id' => $user_id]);
        
        if (!$user_id || $user_id <= 0) {
            Logger::log("❌ Invalid user_id from middleware", [
                'user_id' => $user_id,
                'type' => 'authentication_error'
            ]);
            
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'error' => 'Authentication required'
            ]);
            return;
        }

        // Get pagination parameters
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = ($page - 1) * $limit;
        
        Logger::log("📊 Pagination parameters", [
            'page' => $page,
            'limit' => $limit,
            'offset' => $offset
        ]);

        // Get user's location and preferences
        Logger::log("👤 Getting current user data for ID: $user_id");
        $current_user = $this->discoveryModel->getCurrentUser($user_id);
        
        if (!$current_user) {
            Logger::log("❌ User not found in database", [
                'user_id' => $user_id,
                'type' => 'user_not_found'
            ]);
            
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'error' => 'User not found'
            ]);
            return;
        }

        Logger::log("✅ Current user found", [
            'user_id' => $current_user['id'],
            'name' => $current_user['name'],
            'gender' => $current_user['gender'],
            'preference' => $current_user['preference']
        ]);

        // Get discovery users
        Logger::log("🔍 Querying discovery users...");
        $users = $this->discoveryModel->getDiscoveryUsers(
            $user_id, 
            $current_user,
            $offset, 
            $limit
        );

        // Get total count for pagination
        $total_users = $this->discoveryModel->getDiscoveryUsersCount(
            $user_id, 
            $current_user
        );

        Logger::log("✅ Discovery query complete", [
            'users_found' => count($users),
            'total_users' => $total_users
        ]);

        // ALWAYS return JSON, even if empty
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'users' => $users,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($total_users / $limit),
                'total_users' => $total_users,
                'has_more' => ($offset + count($users)) < $total_users
            ]
        ]);

    } catch (Exception $e) {
        Logger::logError("❌ Exception in getDiscoverUsers", $e);
        
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to get discovery users'
        ]);
    }
}

  public function handleSwipe() {
    error_log("🟢 Handling swipe action...");
    
    try {
        $user_id = jwtMiddleware();
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($input['target_user_id']) || !isset($input['action'])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'Target user ID and action are required'
            ]);
            return;
        }

        $target_user_id = (int)$input['target_user_id'];
        $action = $input['action']; // 'like', 'dislike', or 'super_like'
        
        if (!in_array($action, ['like', 'dislike', 'super_like'])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'error' => 'Invalid action. Must be "like", "dislike", or "super_like"'
            ]);
            return;
        }

        error_log("🔄 Processing swipe - User: $user_id, Target: $target_user_id, Action: $action");
        
        // Record the swipe
        $swipe_result = $this->discoveryModel->recordSwipe($user_id, $target_user_id, $action);
        
        if (!$swipe_result) {
            throw new Exception('Failed to record swipe in database');
        }

        // Check for match if it's a like or super_like
        $is_match = false;
        if ($action === 'like' || $action === 'super_like') {
            $is_match = $this->discoveryModel->checkForMatch($user_id, $target_user_id);
            if ($is_match) {
                error_log("🎉 Match found between $user_id and $target_user_id");
            }
        }

        // Update last active time
        $this->discoveryModel->updateLastActive($user_id);

        http_response_code(200);
        echo json_encode([
            'success' => true,
            'match' => $is_match,
            'message' => $is_match ? 'It\'s a match!' : 'Swipe recorded'
        ]);

    } catch (Exception $e) {
        error_log("❌ Error handling swipe: " . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'Failed to process swipe',
            'message' => $e->getMessage()
        ]);
    }
}

    public function updateLocation() {
        error_log("🟢 Updating user location...");
        
        try {
            $user_id = jwtMiddleware();
            $input = json_decode(file_get_contents('php://input'), true);
            
            if (!isset($input['latitude']) || !isset($input['longitude'])) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'error' => 'Latitude and longitude are required'
                ]);
                return;
            }

            $latitude = (float)$input['latitude'];
            $longitude = (float)$input['longitude'];

            error_log("📍 Updating location for user $user_id: $latitude, $longitude");
            
            $result = $this->discoveryModel->updateUserLocation($user_id, $latitude, $longitude);
            
            if ($result) {
                error_log("✅ Location updated for user: $user_id");
                http_response_code(200);
                echo json_encode([
                    'success' => true,
                    'message' => 'Location updated successfully'
                ]);
            } else {
                throw new Exception('Database update failed');
            }

        } catch (Exception $e) {
            error_log("❌ Error updating location: " . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Failed to update location',
                'message' => $e->getMessage()
            ]);
        }
    }

    public function getMatches() {
        error_log("🟢 Getting user matches...");
        
        try {
            $user_id = jwtMiddleware();
            $matches = $this->discoveryModel->getUserMatches($user_id);

            error_log("✅ Found " . count($matches) . " matches for user: $user_id");
            
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'matches' => $matches
            ]);

        } catch (Exception $e) {
            error_log("❌ Error getting matches: " . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => 'Failed to get matches',
                'message' => $e->getMessage()
            ]);
        }
    }

    public function debug() {
        error_log("🟢 Debug endpoint accessed");
        
        try {
            $user_id = jwtMiddleware();
            
            $current_user = $this->discoveryModel->getCurrentUser($user_id);
            
            // Count total users in database
            $count_query = "SELECT COUNT(*) as total_users FROM users WHERE profile_complete = 1";
            $stmt = $this->db->prepare($count_query);
            $stmt->execute();
            $total_users = $stmt->fetch(PDO::FETCH_ASSOC);
            
            http_response_code(200);
            echo json_encode([
                'success' => true,
                'debug' => [
                    'user_id' => $user_id,
                    'current_user' => $current_user,
                    'total_users_complete' => $total_users['total_users'],
                    'server_time' => date('Y-m-d H:i:s'),
                    'request_method' => $_SERVER['REQUEST_METHOD'],
                    'query_params' => $_GET
                ]
            ]);

        } catch (Exception $e) {
            error_log("❌ Debug error: " . $e->getMessage());
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
?>